unit IWTreeview;

{PUBDIST}

interface

uses
  {$IFDEF VSNET}System.ComponentModel, System.Drawing,{$ENDIF}
  {$IFDEF VSNET}
  {$ELSE}
  {$IFDEF Linux}QControls,{$ELSE}Controls,{$ENDIF}
  {$IFDEF Linux}QForms,{$ELSE}Forms,{$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics,{$ELSE}Graphics,{$ENDIF}
  HTTPApp, Classes, SysUtils,
  IWControl, IWHTMLTag, IWFileReference, IWFont, IWGlobal, IWRenderContext,
  IWBaseInterfaces, IWMarkupLanguageTag;

Type
  TIWTreeViewItems = class;
  TIWTreeView = class;

  IIWTreeViewDesigner = interface
    ['{858EF915-9518-491D-833E-CA313E15A003}']
    procedure RefreshItems;

    procedure ItemsFreed;
  end;

  TIWTreeViewImages = class(TPersistent)
  protected
    FClosedFolderImage: TIWFileReference;
    FOpenFolderImage: TIWFileReference;
    FDocumentImage: TIWFileReference;
    FPlusImage: TIWFileReference;
    FMinusImage: TIWFileReference;
    //
    procedure SetClosedFolderImage(const Value: TIWFileReference);
    procedure SetDocumentImage(const Value: TIWFileReference);
    procedure SetMinusImage(const Value: TIWFileReference);
    procedure SetOpenFolderImage(const Value: TIWFileReference);
    procedure SetPlusImage(const Value: TIWFileReference);
  public
    constructor Create;
    destructor Destroy; override;

    procedure Assign(Source: TPersistent); override;
  published
    property OpenFolderImage: TIWFileReference read FOpenFolderImage write SetOpenFolderImage;
    property ClosedFolderImage: TIWFileReference read FClosedFolderImage write SetClosedFolderImage;
    property DocumentImage: TIWFileReference read FDocumentImage write SetDocumentImage;
    property PlusImage: TIWFileReference read FPlusImage write SetPlusImage;
    property MinusImage: TIWFileReference read FMinusImage write SetMinusImage;
  end;

  TIWTreeViewItem = class(TCollectionItem)
  private
    procedure SetExpanded(const Value: Boolean);
    procedure SetHint(const Value: String);
    procedure SetOnClick(const Value: TNotifyEvent);
  protected
    FURL: String;
    FItemImages: TIWTreeViewImages;
    FIWFont: TIWFont;
    FParentItem: TIWTreeViewItem;
    FCaption: String;
    FExpanded: Boolean;
    FSubItems: TList;
    FName: String;
    FOnClick: TNotifyEvent;
    FTreeView: TIWTreeView;
    FVersion: Integer;
    FHint: String;
    FTag: integer;
    //
    {$IFNDEF VSNET}
    procedure DefineProperties(AFiler: TFiler); override;
    {$ENDIF}
    function GetDisplayName: string; override;
    procedure ReadData(AReader: TReader);
    procedure SetDisplayName(const AValue: string); override;
    procedure SetCaption(const Value: string);
    {$IFNDEF VSNET}
    procedure WriteData(AWriter: TWriter);
    {$ENDIF}
    procedure SetURL(const Value: String);
    procedure SetItemImages(const Value: TIWTreeViewImages);
    procedure SetIWFont(const Value: TIWFont);
    procedure SetParentItem(const Value: TIWTreeViewItem);
    procedure SetName(const Value: string);

    function RenderItem(AIdent: Integer; AVisible: Boolean; AAllClickable: Boolean): TIWHTMLTag;

    function GetPlusImageLocation: String;
    function GetMinusImageLocation: String;
    function GetClosedFolderImageLocation: String;
    function GetDocumentImageLocation: String;
    function GetOpenFolderImageLocation: String;

    procedure SetItemVersion(const Value: Integer);

    procedure AssignTo(ADest: TPersistent); override;

    property PlusImageLocation: String read GetPlusImageLocation;
    property MinusImageLocation: String read GetMinusImageLocation;
    property OpenFolderImageLocation: String read GetOpenFolderImageLocation;
    property ClosedFolderImageLocation: String read GetClosedFolderImageLocation;
    property DocumentImageLocation: String read GetDocumentImageLocation;

    property TreeView: TIWTreeView read FTreeView write FTreeView;
  public
    constructor Create(ACollection: TCollection); override;
    destructor Destroy; override;

    procedure DoClick;
    function GetNamePath: string; override;
    function HasChild: Boolean;

    property ParentItem: TIWTreeViewItem read FParentItem write SetParentItem;
    property SubItems: TList read FSubItems;
  published

    property Caption: string read FCaption write SetCaption;
    property Expanded: Boolean read FExpanded write SetExpanded;
    property Font: TIWFont read FIWFont write SetIWFont;
    property Hint: String read FHint write SetHint;
    property ItemImages: TIWTreeViewImages read FItemImages write SetItemImages;
    property Name: string read FName write SetName;
    property Tag: integer read FTag write FTag;
    property URL: String read FURL write SetURL;
    property Version: Integer read FVersion write SetItemVersion;

    property OnClick: TNotifyEvent read FOnClick write SetOnClick;
  end;

  TIWTreeViewItems = class(TOwnedCollection)
  protected
    FDesigner: IIWTreeViewDesigner;

    function GetItem(AIndex: Integer): TIWTreeViewItem;
    procedure SetItem(AIndex: Integer; const AValue: TIWTreeViewItem);

    function RenderTreeView(AAllClickable: Boolean): TIWHTMLTagCollection;
  public
    function Add: TIWTreeViewItem;

    constructor Create(AOwner: TPersistent);
    destructor Destroy; override;

    procedure SetDesigner(ADesigner: IIWTreeViewDesigner);

    property Items[AIndex: Integer]: TIWTreeViewItem read GetItem write SetItem; default;
  end;

  TIWTreeViewClickEvent = procedure(Sender: TObject; ATreeViewItem: TIWTreeViewItem) of object;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWTreeView.bmp}
  [ToolboxItem(false), ToolboxBitmap(typeof(TIWTreeView), 'TIWTreeView.bmp')]
  {$ENDIF}
  TIWTreeView = class(TIWCustomControl, IIWInputControl, IIWSubmitControl)
  private
    FContext: TIWComponent40Context;
  protected
    FUniqueCounter : integer;
    FSubmitParam : String;
    FTreeNodes: String;
    FTreeViewItems: TIWTreeViewItems;
    FTreeViewImages: TIWTreeViewImages;
    // FVertScrollBarVisible: Boolean;
    // FHorScrollBarVisible: Boolean;
    FTreeViewItem: TIWTreeViewItem;

    FOnClick: TIWTreeViewClickEvent;
    //
    function GetUniqueName: string;
    //
    procedure DefineProperties(AFiler: TFiler); override;
    procedure ReadData(AReader: TReader);
    procedure WriteData(AWriter: TWriter);

    procedure SetTreeViewItems(const AValue: TIWTreeViewItems);

    function GetPlusImageLocation: String;
    function GetMinusImageLocation: String;
    function GetClosedFolderImageLocation: String;
    function GetDocumentImageLocation: String;
    function GetOpenFolderImageLocation: String;

    procedure SetValue(const AValue: string); virtual;
    procedure Submit(const AValue: string); override;

    procedure Loaded; override;

    procedure InitControl; override;
    property PlusImageLocation: String read GetPlusImageLocation;
    property MinusImageLocation: String read GetMinusImageLocation;
    property OpenFolderImageLocation: String read GetOpenFolderImageLocation;
    property ClosedFolderImageLocation: String read GetClosedFolderImageLocation;
    property DocumentImageLocation: String read GetDocumentImageLocation;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function GetSubmitParam : String;

    property Items: TIWTreeViewItems read FTreeViewItems write SetTreeViewItems;
    property Selected: TIWTreeViewItem read FTreeViewItem write FTreeViewItem;
  published
    property FriendlyName;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    // this property is read/write becouse by default IDE does not show read only properties
    property TreeNodes: String read FTreeNodes write FTreeNodes; // this is here to allow to edit Items via the Tree nodes editor
    property TreeViewImages: TIWTreeViewImages read FTreeViewImages write FTreeViewImages;
    property UseFrame;
    property TreeItemClick: TIWTreeViewClickEvent read FOnClick write FOnClick;
  end;

implementation

uses
  IWApplication, IWAppForm, IWTypes, IWForm, IWServer,
  SWSystem, SWStrings, IWBaseControl;

{ TIWTreeView }

procedure TIWTreeView.InitControl;
begin
  inherited;
  Width := 200;
  Height := 200;
  FTreeViewItems := TIWTreeViewItems.Create(self);
  FTreeViewImages := TIWTreeViewImages.Create;
  set_RenderSize(true);

  SupportsPartial := false;
  // FVertScrollBarVisible := false;
  // FHorScrollBarVisible := false;
end;

function TIWTreeView.GetUniqueName : string;
begin
  FUniqueCounter := FUniqueCounter + 1;
  Result := 'i' + IntToStr(FUniqueCounter);
end;

procedure TIWTreeView.DefineProperties(AFiler: TFiler);
begin
  inherited DefineProperties(AFiler);
  AFiler.DefineProperty('Items', ReadData, WriteData, True);
end;

procedure TIWTreeView.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FTreeViewItems);
  FreeAndNil(FTreeViewImages);
  inherited;
end;

function TIWTreeView.GetClosedFolderImageLocation: String;
begin
  result := iif(FTreeViewImages.ClosedFolderImage.Location(GServerController.FilesURL),
    FTreeViewImages.ClosedFolderImage.Location(GServerController.FilesURL),
    FContext.WebApplication.InternalURLBase + '/gfx/TreeView_closed.gif');
end;

function TIWTreeView.GetDocumentImageLocation: String;
begin
  result := iif(FTreeViewImages.DocumentImage.Location(GServerController.FilesURL),
    FTreeViewImages.DocumentImage.Location(GServerController.FilesURL),
    FContext.WebApplication.InternalURLBase + '/gfx/TreeView_document.gif');
end;

function TIWTreeView.GetMinusImageLocation: String;
begin
  result := iif(FTreeViewImages.MinusImage.Location(GServerController.FilesURL),
    FTreeViewImages.MinusImage.Location(GServerController.FilesURL),
    FContext.WebApplication.InternalURLBase + '/gfx/TreeView_minus.gif');
end;

function TIWTreeView.GetOpenFolderImageLocation: String;
begin
  result := iif(FTreeViewImages.OpenFolderImage.Location(GServerController.FilesURL),
    FTreeViewImages.OpenFolderImage.Location(GServerController.FilesURL),
    FContext.WebApplication.InternalURLBase + '/gfx/TreeView_open.gif');
end;

function TIWTreeView.GetPlusImageLocation: String;
begin
  result := iif(FTreeViewImages.PlusImage.Location(GServerController.FilesURL),
    FTreeViewImages.PlusImage.Location(GServerController.FilesURL),
    FContext.WebApplication.InternalURLBase + '/gfx/TreeView_plus.gif');
end;

procedure TIWTreeView.Loaded;
begin
  inherited Loaded;
end;

procedure TIWTreeView.ReadData(AReader: TReader);
begin
  AReader.CheckValue(vaCollection);
  AReader.ReadCollection(Items);
end;

function TIWTreeView.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWTreeView.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LScript: string;

  function BuildAddition(AItems: TList): String;
  var
    i: Integer;
    LItem: TIWTreeViewItem;
    LIdent: Integer;
    LItems: TList;
  begin
    result := '';
    LItems := AItems;
    if AItems = nil then begin
      LItems := TList.Create;
      for i := 0 to Items.Count - 1 do begin
        if Items[i].ParentItem = nil then begin
          LItems.Add(Items[i]);
        end;
      end;
    end;

    try
      for i := 0 to LItems.Count - 1 do begin
        LIdent := 1;
        LItem := TIWTreeViewItem(LItems.Items[i]);
        while LItem.ParentItem <> nil do begin
          Inc(LIdent);
          LItem := LItem.ParentItem;
        end;
        LItem := TIWTreeViewItem(LItems.Items[i]);
        Result := result + StringOfChar(' ', LIdent * 2)
         + 'new ' + iif(AContext.WebApplication.IsPartialUpdate, 'IWTop().') + 'AddTVItem('
         + iif(AContext.WebApplication.IsPartialUpdate, 'IWTop().') + HTMLName + 'IWCL'
         + ', "' + LItem.Name + '"'
         // + ', "' + LItem.Caption + '"'
         // + ', ' + iif(Assigned(LItem.OnClick), IntToStr(Integer(LItem)), '0')
         // + ', "' + LItem.Font.CSSStyle + '"'
         // + ', ' + iif(LItem.Expanded, 'true', 'false')
         + ', ';

        if LItem.FSubItems.Count > 0 then begin
          result := result + EOL + BuildAddition(LItem.FSubItems) + ', ';
        end else begin
          result := result + 'null, ';
        end;

        if i = LItems.Count - 1 then begin
          result := result + 'null';
        end else begin
          result := result + EOL;
        end;
      end;
      Result := Result + StringOfChar(')', LItems.Count);
    finally
      if AItems = nil then
        LItems.Free;
    end;
  end;

Var
  // LStyle: String;
  LInputTag: TIWHTMLTag;
  LDivTag: TIWHTMLTag;
  LStatus: String;
  i: Integer;
begin
  FContext := TIWComponent40Context(AContext);

  set_Clip(UseFrame);

  if Items.Count = 0 then begin
    Result := nil;
  end else begin
    FContext.AddScriptFile('/js/IWTree.js');

    LScript := iif(FContext.WebApplication.IsPartialUpdate, 'window.parent.')
      + HTMLName + 'IWCL.Items = ' + EOL + BuildAddition(nil) + ';' + EOL
      + iif(FContext.WebApplication.IsPartialUpdate, 'window.parent.')
      + 'InitTreeView('+ iif(FContext.WebApplication.IsPartialUpdate, 'window.parent.') + HTMLName + 'IWCL, "' + HTMLName + 'Container");' + EOL;

    with FContext do begin
      {CacheImage(Self.HTMLName + '_Open', LOpenImg);
      CacheImage(Self.HTMLName + '_Closed', LCloseImg);
      CacheImage(Self.HTMLName + '_Document', LDocImg);
      CacheImage(Self.HTMLName + '_Plus', LPlusImg);
      CacheImage(Self.HTMLName + '_Minus', LMinusImg);}

      CacheImage(Self.HTMLName + '_Blank', WebApplication.InternalURLBase + '/gfx/Tp.gif');
    end;
    if not FContext.WebApplication.IsPartialUpdate then
      FContext.AddToInitProc(LScript)
    else
      FContext.AddToUpdateInitProc(LScript);

    LDivTag := TIWHTMLTag.CreateTag('DIV'); try
      LDivTag.Contents.AddTagCollection(FTreeViewItems.RenderTreeView(Assigned(TreeItemClick)), true);
      LDivTag.AddStringParam('id', HTMLName + 'Container');
      LDivTag.AddStringParam('style', 'position:relative; left:0; top:0; width:4096; height:4096;');

      LStatus := '';
      for i := 0 to Items.Count -1 do begin
        if Items[i].Expanded then
          LStatus := LStatus + ',' + Items[i].Name + ',';
      end;

      LInputTag := TIWHTMLTag.CreateTag('input'); try
        LInputTag.AddStringParam('type', 'hidden');
        LInputTag.AddStringParam('name', HTMLName);
        if not UseFrame then
          LInputTag.AddStringParam('ID', HTMLName);
        LInputTag.AddStringParam('value', HTTPEncode(LStatus));
      except FreeAndNil(LInputTag); raise; end;
    except FreeAndNil(LDivTag); raise; end;

    Result := TIWHTMLTag.CreateTag('DIV'); try
      Result.Contents.AddTagAsObject(LInputTag);
      Result.Contents.AddTagAsObject(LDivTag);
      {LStyle := ''; //'background-color:#808080;';
      if VertScrollBarVisible then begin
        LStyle := LStyle + 'overflow-y:auto;';
      end;

      if HorScrollBarVisible then begin
        LStyle := LStyle + 'overflow-x:auto;';
      end;

      if LStyle <> '' then begin
        Result.AddStringParam('style', LStyle);
      end;}
    except
      FreeAndNil(LInputTag);
      FreeAndNil(LDivTag);
      FreeAndNil(Result);
      raise;
    end;
  end;
end;

procedure TIWTreeView.SetTreeViewItems(const AValue: TIWTreeViewItems);
begin
  FTreeViewItems.Assign(AValue);
end;

procedure TIWTreeView.SetValue(const AValue: string);
Var
  LStatusValue: String;
  LItemName: String;
  i: Integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  LStatusValue := HTTPDecode(AValue);
  if Trim(LStatusValue) <> '' then begin
    for i := 0 to Items.Count - 1 do begin
      Items[i].Expanded := false;
    end;

    lItemName := '';
    while LStatusValue <> '' do begin
      while (LItemName = ',') or (LItemName = '') do begin
        LItemName := Fetch(LStatusValue, ',');
      end;
      for i := 0 to Items.Count - 1 do begin
        if Items[i].Name = LItemName then begin
          Items[i].Expanded := true;
          break;
        end;
      end;
      LItemName := '';
    end;
  end;
end;

procedure TIWTreeView.Submit(const AValue: string);
Var
  i: Integer;
begin
  FSubmitParam := AValue;
  FTreeViewItem := nil;
  for i := 0 to Items.Count - 1 do begin
    if Items[i].Name = AValue then begin
      FTreeViewItem := Items[i];
      Items[i].DoClick;
      if Assigned(FOnClick) then begin
        FOnClick(self, Items[i]);
      end;
      break;
    end;
  end;
end;

procedure TIWTreeView.WriteData(AWriter: TWriter);
begin
  AWriter.WriteCollection(Items);
end;

{ TIWTreeViewItem }

procedure TIWTreeViewItem.AssignTo(ADest: TPersistent);
begin
  if ADest is TIWTreeViewItem then begin
    with TIWTreeViewItem(ADest) do begin
      Caption := Self.Caption;
      Expanded := Self.Expanded;
      ParentItem := Self.ParentItem;
      Name := Self.Name;
      URL := Self.URL;
      OnClick := Self.OnClick;
      ItemImages.Assign(Self.ItemImages);
    end;
  end else begin
    inherited AssignTo(ADest);
  end;
end;

constructor TIWTreeViewItem.Create(ACollection: TCollection);
var
//  i: Integer;
  LItem: TIWTreeViewItem;
  LName: string;
  LTree: TIWTreeView;
  LOwner: TPersistent;
begin
  // to eliminate warning
  LOwner := nil;
  inherited Create(ACollection);
  // Must be before Find Name as it uses Items
  FSubItems := TList.Create;
  // Find a unique name
  //
  // Travel up to find the tree itself
  LTree := nil;
  LItem := Self;
  while LTree = nil do begin
    LOwner := LItem.GetOwner;
    if LOwner is TIWTreeView then begin
      LTree := TIWTreeView(LOwner);
    end else begin
      {$IFDEF VCL8ORABOVE}
      if LOwner is TIWTreeViewItem then begin
        LItem := TIWTreeViewItem(LOwner);
      end else begin
        LTree := TIWTreeView(TIWTreeViewItems( LOwner ).Owner);
      end;
      {$ELSE}
      LItem := TIWTreeViewItem(LOwner);
      {$ENDIF}
    end;
  end;
  // Find a unique name
  LName := LTree.GetUniqueName; // by DBR, bug #2051
  //
  FName := LName;
  // Remove the below since at run-time it renames the caption back to the default
  if FCaption = '' then begin
    FCaption := LName;
  end;

  FItemImages := TIWTreeViewImages.Create;
  FIWFont := TIWFont.Create;

  FTreeView := TIWTreeView(TIWTreeViewItems(Collection).getOwner);

  FExpanded := True;
  FVersion := 0;
end;

{$IFNDEF VSNET}
procedure TIWTreeViewItem.DefineProperties(AFiler: TFiler);
begin
  inherited DefineProperties(AFiler);
  case FVersion of
    0: begin
      AFiler.DefineProperty('SubItems', ReadData, WriteData, True);
    end;
    1: begin
      AFiler.DefineProperty('ParentItem', ReadData, WriteData, True);
    end;
  end;
end;
{$ENDIF}

destructor TIWTreeViewItem.Destroy;
begin
  ParentItem := nil;
  if Assigned(FSubItems) then begin
    while FSubItems.Count > 0 do begin
      TObject(FSubItems[0]).Free;
    end;
  end;
  FreeAndNil(FSubItems);
  FreeAndNil(FItemImages);
  FreeAndNil(FIWFont);
  inherited Destroy;
end;

procedure TIWTreeViewItem.DoClick;
begin
  if Assigned(FOnClick) then begin
    FOnClick(self);
  end;
end;

function TIWTreeViewItem.GetClosedFolderImageLocation: String;
begin
  if TreeView <> nil then begin
    result := ItemImages.ClosedFolderImage.Location(GServerController.FilesURL);
    if result = '' then begin
      result := TreeView.ClosedFolderImageLocation;
    end;
  end else begin
    result := '';
  end;
end;

function TIWTreeViewItem.GetDisplayName: string;
begin
  Result := FName;
  Version := 1;
end;

function TIWTreeViewItem.GetDocumentImageLocation: String;
begin
  if TreeView <> nil then begin
    result := ItemImages.DocumentImage.Location(GServerController.FilesURL);
    if result = '' then begin
      result := TreeView.DocumentImageLocation;
    end;
  end else begin
    result := '';
  end;
end;

function TIWTreeViewItem.GetMinusImageLocation: String;
begin
  if TreeView <> nil then begin
    result := ItemImages.MinusImage.Location(GServerController.FilesURL);
    if result = '' then begin
      result := TreeView.MinusImageLocation;
    end;
  end else begin
    result := '';
  end;
end;

function TIWTreeViewItem.GetNamePath: string;
begin
  if Collection <> nil then begin
    Result := TIWTreeViewItems(Collection).GetOwner.GetNamePath + '.' + GetDisplayName;
  end else begin
    Result := inherited GetNamePath;
  end;
end;

procedure TIWTreeViewItem.ReadData(AReader: TReader);
Var
  AIndex: Integer;
  LSubItems: TIWTreeViewItems;
  LItem: TIWTreeViewItem;
  i: Integer;
begin
  case FVersion of
    0: begin
      AReader.CheckValue(vaCollection);
      LSubItems := TIWTreeViewItems.Create(TIWTreeViewItems(GetOwner).GetOwner); try
        AReader.ReadCollection(LSubItems);
        for AIndex := 0 to LSubItems.Count - 1 do begin
          LItem := TIWTreeViewItems(GetOwner).Add;
          LItem.Assign(LSubItems[AIndex]);

          for i := 0 to LSubItems[AIndex].SubItems.Count - 1 do begin
            TIWTreeViewItem(LSubItems[AIndex].SubItems.Items[i]).ParentItem := LItem;
          end;

          if LItem.ParentItem = nil then begin
            LItem.ParentItem := self;
          end;
        end;
      finally
        LSubItems.Free;
      end;
      FVersion := 1;
      {LDesigner := FindRootDesigner(self);
      if LDesigner <> nil then begin
        LDesigner.Modified;
      end;}
    end;
    1: begin
      AIndex := AReader.ReadInteger; // StrToIntDef(AReader.ReadStr, -1); //
      if AIndex = -1 then begin
        ParentItem := nil;
      end else begin
        ParentItem := TreeView.Items[AIndex];
      end;
    end;
  end;
end;

function TIWTreeViewItem.GetOpenFolderImageLocation: String;
begin
  if TreeView <> nil then begin
    result := ItemImages.OpenFolderImage.Location(GServerController.FilesURL);
    if result = '' then begin
      result := TreeView.OpenFolderImageLocation;
    end;
  end else begin
    result := '';
  end;
end;

function TIWTreeViewItem.GetPlusImageLocation: String;
begin
  if TreeView <> nil then begin
    result := ItemImages.PlusImage.Location(GServerController.FilesURL);
    if result = '' then begin
      result := TreeView.PlusImageLocation;
    end;
  end else begin
    result := '';
  end;
end;

function TIWTreeViewItem.RenderItem(AIdent: Integer; AVisible: Boolean; AAllClickable: Boolean): TIWHTMLTag;
Var
  i: Integer;
  LTag: TIWHTMLTag;
begin
  result := TIWHTMLTag.CreateTag('SPAN');
  result.AddStringParam('id', Name);
  result.AddStringParam('class', Font.CSSStyle);
  if not AVisible then begin
    result.AddStringParam('style', 'visibility:hidden; display:none;');
  end;
  if AIdent > 0 then
    with result.Contents.addTag('img') do begin
      IsPreformated := true;
      AddIntegerParam('border', 0);
      AddStringParam('src', TreeView.FContext.WebApplication.InternalURLBase + '/gfx/Tp.gif');
      AddIntegerParam('width', 16 * AIdent);
      AddIntegerParam('height', 16);
    end;

  if FSubItems.Count > 0 then begin
    with Result.Contents.AddTag('a') do begin
      IsPreformated := true;
      AddStringParam('href', 'javascript:' + TreeView.HTMLName + 'IWCL.OpenTreeItem(''' + Name + ''');');
      AddStringParam('id', 'plusImg_' + Name);
      if Expanded then begin
        AddStringParam('style', 'visibility:hidden; display:none;');
      end;

      with Contents.AddTag('img') do begin
        AddIntegerParam('border', 0);
        AddStringParam('src', PlusImageLocation);
        AddIntegerParam('width', 16);
        AddIntegerParam('height', 16);
      end;
      with Contents.addTag('img') do begin
        AddIntegerParam('border', 0);
        AddStringParam('src', ClosedFolderImageLocation);
        AddIntegerParam('width', 16);
        AddIntegerParam('height', 16);
      end;
    end;

    with Result.Contents.AddTag('a') do begin
      IsPreformated := true;
      AddStringParam('href', 'javascript:' + TreeView.HTMLName + 'IWCL.CloseTreeItem(''' + Name + ''');');
      AddStringParam('id', 'minusImg_' + Name);
      if not Expanded then begin
        AddStringParam('style', 'visibility:hidden; display:none;');
      end;

      with Contents.AddTag('img') do begin
        AddIntegerParam('border', 0);
        AddStringParam('src', MinusImageLocation);
        AddIntegerParam('width', 16);
        AddIntegerParam('height', 16);
      end;

      with Contents.addTag('img') do begin
        AddIntegerParam('border', 0);
        AddStringParam('src', OpenFolderImageLocation);
        AddIntegerParam('width', 16);
        AddIntegerParam('height', 16);
      end;
    end;
  end else begin
    with Result.Contents.AddTag('img') do begin
      AddIntegerParam('border', 0);
      AddStringParam('src', TreeView.FContext.WebApplication.InternalURLBase + '/gfx/Tp.gif');
      AddIntegerParam('width', 16);
      AddIntegerParam('height', 16);
    end;

    with Result.Contents.addTag('img') do begin
      AddIntegerParam('border', 0);
      AddStringParam('src', DocumentImageLocation);
      AddIntegerParam('width', 16);
      AddIntegerParam('height', 16);
    end;
  end;

  if AAllClickable or Assigned(OnClick) or (Trim(FURL) <> '') then begin
    LTag := result.Contents.AddTag('a');
  end else begin
    LTag := result.Contents.AddTag('span');
  end;
  with LTag do begin
    if (Trim(FURL) = '') and (Assigned(OnClick) or AAllClickable) then begin
      AddStringParam('href', 'javascript:SubmitClick(''' + TreeView.HTMLName + ''', ''' + Name + ''');');
      AddStringParam('onclick', 'SubmitClick(''' + TreeView.HTMLName + ''', ''' + Name + '''); return false;');
    end
    else begin
      AddStringParam('href', FURL);
    end;
    IsPreformated := true;
    if Trim(FHint) <> '' then begin
      AddStringParam('Title', StringReplace(FHint, '''', '&#39', [rfReplaceAll]));
    end;
    if Font.CSSStyle <> '' then begin
      AddStringParam('class', Font.CSSStyle);
    end else if Font.Enabled then begin
      AddStringParam('style',Font.FontToStringStyle(TreeView.FContext.WebApplication.Browser));
    end;
    Contents.AddText(Caption);
  end;
  result.Contents.AddText('<br>');
  for i := 0 to SubItems.Count - 1 do begin
    result.Contents.AddTagAsObject(TIWTreeViewItem(SubItems.Items[i]).RenderItem(AIdent + 1, Expanded and AVisible, AAllClickable));
  end;
end;

procedure TIWTreeViewItem.SetCaption(const Value: string);
Var
  LDesigner: IDesignerNotify;
  LCollection: TCollection;
begin
  FCaption := Value;
  LDesigner := FindRootDesigner(self);
  if LDesigner <> nil then begin
    LDesigner.Modified;
  end;

  LCollection := TCollection(GetOwner);
  if (LCollection <> nil) and (LCollection is TIWTreeViewItems) then begin
    if TIWTreeViewItems(LCollection).FDesigner <> nil then begin
      TIWTreeViewItems(LCollection).FDesigner.RefreshItems;
    end;
  end;
end;

procedure TIWTreeViewItem.SetDisplayName(const AValue: string);
begin
  FName := AValue;
  inherited SetDisplayName(AValue);
end;

procedure TIWTreeViewItem.SetItemImages(const Value: TIWTreeViewImages);
begin
  FItemImages.Assign(Value);
end;

procedure TIWTreeViewItem.SetIWFont(const Value: TIWFont);
begin
  FIWFont.Assign(Value);
end;

procedure TIWTreeViewItem.SetParentItem(const Value: TIWTreeViewItem);
begin
  if Assigned(FParentItem) then begin
    FParentItem.SubItems.Delete(FParentItem.SubItems.IndexOf(Self));
  end;

  FParentItem := Value;

  if Assigned(FParentItem) then begin
    FParentItem.SubItems.Add(Self);
  end;

  Version := 1;
end;

procedure TIWTreeViewItem.SetURL(const Value: String);
begin
  FURL := Value;
end;

{$IFNDEF VSNET}
procedure TIWTreeViewItem.WriteData(AWriter: TWriter);
Var
  AIndex: Integer;
  vt: TValueType;
begin
  if not Assigned(FParentItem) then begin
    AIndex := -1;
  end else begin
    AIndex := FParentItem.Index;
  end;
  vt := vaInt32;
  AWriter.Write(Byte(vt), sizeOf(vt));

  AWriter.Write(AIndex, SizeOf(AIndex));
end;
{$ENDIF}

procedure TIWTreeViewItem.SetItemVersion(const Value: Integer);
begin
  if FVersion = 0 then begin
    FVersion := Value;
  end;
end;

function TIWTreeViewItem.HasChild: Boolean;
begin
  result := FSubItems.Count > 0;
end;

procedure TIWTreeViewItem.SetName(const Value: string);
begin
  FName := StringReplace(Value, ' ', '_', [rfReplaceAll]);
end;

procedure TIWTreeViewItem.SetExpanded(const Value: Boolean);
begin
  FExpanded := Value;
  TreeView.Invalidate;
end;

procedure TIWTreeViewItem.SetHint(const Value: String);
begin
  FHint := Value;
  TreeView.Invalidate;
end;

procedure TIWTreeViewItem.SetOnClick(const Value: TNotifyEvent);
begin
  FOnClick := Value;
  TreeView.Invalidate;
end;

{ TIWTreeViewItems }

constructor TIWTreeViewItems.Create(AOwner: TPersistent);
begin
  inherited Create(AOwner, TIWTreeViewItem);
end;

function TIWTreeViewItems.Add: TIWTreeViewItem;
begin
  result := TIWTreeViewItem(inherited Add);
  result.TreeView := TIWTreeView(GetOwner);
  result.TreeView.Invalidate;
end;

function TIWTreeViewItems.GetItem(AIndex: Integer): TIWTreeViewItem;
begin
  result := TIWTreeViewItem(inherited Items[AIndex]);
end;

procedure TIWTreeViewItems.SetItem(AIndex: Integer; const AValue: TIWTreeViewItem);
begin
  TIWTreeViewItem(inherited Items[AIndex]).Assign(AValue);
end;

procedure TIWTreeViewItems.SetDesigner(ADesigner: IIWTreeViewDesigner);
begin
  FDesigner := ADesigner;
end;

destructor TIWTreeViewItems.Destroy;
begin
  if FDesigner <> nil then begin
    FDesigner.ItemsFreed;
  end;
  inherited Destroy;
end;

function TIWTreeViewItems.RenderTreeView(AAllClickable: Boolean): TIWHTMLTagCollection;
var
  i: Integer;
begin
  result := TIWHTMLTagCollection.Create(nil, TIWHTMLTag);
  for i := 0 to Count - 1 do begin
    if Items[i].FParentItem = nil then
      // TODO: Check this
      result.AddTagAsObject(Items[i].RenderItem(0, true, AAllClickable))
  end;
end;

{ TIWTreeViewImages }

procedure TIWTreeViewImages.Assign(Source: TPersistent);
Var
  LSource: TIWTreeViewImages;
begin
  if Source is TIWTreeViewImages then begin
    LSource := Source as TIWTreeViewImages;
    FClosedFolderImage.Assign(LSource.FClosedFolderImage);
    FOpenFolderImage.Assign(LSource.FOpenFolderImage);
    FDocumentImage.Assign(LSource.FDocumentImage);
    FPlusImage.Assign(LSource.FPlusImage);
    FMinusImage.Assign(LSource.FMinusImage);
   end else begin
    inherited Assign(Source);
  end;
end;

constructor TIWTreeViewImages.Create;
begin
  inherited Create;

  FClosedFolderImage := TIWFileReference.Create;
  FOpenFolderImage := TIWFileReference.Create;
  FDocumentImage := TIWFileReference.Create;
  FPlusImage := TIWFileReference.Create;
  FMinusImage := TIWFileReference.Create;
end;

destructor TIWTreeViewImages.Destroy;
begin
  FreeAndNil(FClosedFolderImage);
  FreeAndNil(FOpenFolderImage);
  FreeAndNil(FDocumentImage);
  FreeAndNil(FPlusImage);
  FreeAndNil(FMinusImage);

  inherited Destroy;
end;

procedure TIWTreeViewImages.SetClosedFolderImage(
  const Value: TIWFileReference);
begin
  FClosedFolderImage.Assign(Value);
end;

procedure TIWTreeViewImages.SetDocumentImage(
  const Value: TIWFileReference);
begin
  FDocumentImage.Assign(Value);
end;

procedure TIWTreeViewImages.SetMinusImage(const Value: TIWFileReference);
begin
  FMinusImage.Assign(Value);
end;

procedure TIWTreeViewImages.SetOpenFolderImage(
  const Value: TIWFileReference);
begin
  FOpenFolderImage.Assign(Value);
end;

procedure TIWTreeViewImages.SetPlusImage(const Value: TIWFileReference);
begin
  FPlusImage.Assign(Value);
end;

initialization
  TIWServer.AddInternalFile('IW_JS_IWTree', '/js/IWTree.js');
  TIWServer.AddInternalFile('IW_GFX_TreeView_Open', '/gfx/TreeView_open.gif');
  TIWServer.AddInternalFile('IW_GFX_TreeView_Closed', '/gfx/TreeView_closed.gif');
  TIWServer.AddInternalFile('IW_GFX_TreeView_Document', '/gfx/TreeView_document.gif');
  TIWServer.AddInternalFile('IW_GFX_TreeView_Plus', '/gfx/TreeView_plus.gif');
  TIWServer.AddInternalFile('IW_GFX_TreeView_Minus', '/gfx/TreeView_minus.gif');
end.


